
/*
 * Project     : Plantilla
 *
 * CVS Id      : $Id: GuiaPDFEvents.java,v 1.30 2022/11/03 08:46:52 corhuot Exp $
 *
 * (C) 2005 PROFit, S.A. All rights reserved.
 * 
 * Construye, a partir de un objeto Horario, un PDF con ese horario representado.
 * 
 * GUIADOCENTE-106: Migracion a iText 7
 */

package es.uv.gui.pdf;

import java.io.IOException;
import java.net.MalformedURLException;
import java.util.logging.Logger;

import com.itextpdf.io.font.constants.StandardFonts;
import com.itextpdf.io.image.ImageDataFactory;
import com.itextpdf.kernel.colors.ColorConstants;
import com.itextpdf.kernel.events.Event;
import com.itextpdf.kernel.events.IEventHandler;
import com.itextpdf.kernel.events.PdfDocumentEvent;
import com.itextpdf.kernel.font.PdfFont;
import com.itextpdf.kernel.font.PdfFontFactory;
import com.itextpdf.kernel.geom.Rectangle;
import com.itextpdf.kernel.pdf.PdfDocument;
import com.itextpdf.kernel.pdf.PdfPage;
import com.itextpdf.kernel.pdf.canvas.PdfCanvas;
import com.itextpdf.layout.Canvas;
import com.itextpdf.layout.Document;
import com.itextpdf.layout.borders.Border;
import com.itextpdf.layout.borders.SolidBorder;
import com.itextpdf.layout.element.Cell;
import com.itextpdf.layout.element.Image;
import com.itextpdf.layout.element.Paragraph;
import com.itextpdf.layout.element.Table;
import com.itextpdf.layout.properties.TextAlignment;
import com.itextpdf.layout.properties.UnitValue;
import com.itextpdf.layout.properties.VerticalAlignment;

import es.uv.guia.utiles.ConstantesGuiaDocente;


public class GuiaPDFEvents implements IEventHandler {

    private final Document documento; 
    private static final String ESCUDO= GuiaPDF.sResourcesPath + GuiaConstantes.ESCUDO;
    private static final Logger LOGGER = Logger.getLogger("es.uv.gui.pdf.GuiaPDFEvents");

    public GuiaPDFEvents(Document doc) {
        documento = doc;
    }
    
    /**
     * Crea el rectangulo donde pondremos el encabezado
     * @param docEvent Evento de documento
     * @return Area donde colocaremos el encabezado
     */
    private Rectangle crearRectanguloEncabezado(PdfDocumentEvent docEvent) {
        PdfDocument pdfDoc = docEvent.getDocument();
        PdfPage page = docEvent.getPage();        
        
        float xEncabezado = pdfDoc.getDefaultPageSize().getX() + documento.getLeftMargin();
        float yEncabezado = pdfDoc.getDefaultPageSize().getTop() - documento.getTopMargin();
        float anchoEncabezado = page.getPageSize().getWidth() - 80;
        float altoEncabezado = 90F;
        
        return new Rectangle(xEncabezado, yEncabezado, anchoEncabezado, altoEncabezado);               
    }
    
    /**
     * Crea el rectangulo donde pondremos el pie de pagina
     * @param docEvent Evento del documento
     * @return Area donde colocaremos el pie de pagina
     */
    private Rectangle crearRectanguloPie(PdfDocumentEvent docEvent) {
        PdfDocument pdfDoc = docEvent.getDocument();
        PdfPage page = docEvent.getPage();
        
        float xPie = pdfDoc.getDefaultPageSize().getX() + documento.getLeftMargin();
        float yPie = pdfDoc.getDefaultPageSize().getBottom() ;
        float anchoPie = page.getPageSize().getWidth() - 72;
        float altoPie = 40F;

        return new Rectangle(xPie, yPie, anchoPie, altoPie);
    }
    
    /**
     * Crea la tabla que contendra el mensaje del encabezado
     * @param mensaje Mensaje que desplegaremos
     * @return Tabla con el mensaje de encabezado
     */
    private Table crearTablaEncabezado(String mensaje) { 
        Table tablaEncabezado = new Table(2);
        tablaEncabezado.setWidth(UnitValue.createPercentValue(100));
        PdfFont font = null;
        SolidBorder borde = new SolidBorder(0.1f);
        try {          	
        	Image img = new Image(ImageDataFactory.create(ESCUDO));			
			img.scaleToFit(150, 150);		
			
			font = PdfFontFactory.createFont(StandardFonts.HELVETICA_BOLD);	
			Cell c = new Cell();
		    c.add(img);
		    c.setBorder(Border.NO_BORDER);
		    c.setBorderTop(borde);
		    c.setBorderBottom(borde);
		    c.setPaddingTop(10);
		    c.setPaddingBottom(10);
		    c.setVerticalAlignment(VerticalAlignment.MIDDLE);
			tablaEncabezado.addCell(c);
			
			Paragraph p = new Paragraph();
			p.add(mensaje);			
			Cell c2 = new Cell();
			c2.add(p);
			c2.setBorder(Border.NO_BORDER);
		    c2.setBorderTop(borde);
		    c2.setBorderBottom(borde);
		    c2.setPaddingTop(10);
		    c2.setPaddingBottom(10);
		    c2.setVerticalAlignment(VerticalAlignment.MIDDLE);
		    
			tablaEncabezado.setFont(font);
			tablaEncabezado.setTextAlignment(TextAlignment.RIGHT);
			tablaEncabezado.setFontSize(14.0f);
	        tablaEncabezado.addCell(c2);
	        tablaEncabezado.setPaddingBottom(10);
	        
		} catch (MalformedURLException e) {
			LOGGER.info("Error MalformedURLException al crear la tabla de encabezado: " + e.getMessage());
		} catch (IOException e) {
			LOGGER.info("Error IOException al crear la tabla de encabezado: " + e.getMessage());
		}
        
        return tablaEncabezado;
    }   
    
    /**
     * Crea la tabla de pie de pagina, con el numero de pagina
     * @param docEvent Evento del documento
     * @return Pie de pagina con el numero de pagina
     */
    private Table crearTablaPie(PdfDocumentEvent docEvent) {
        
    	Table tablaPie = null;
		
		try {
			PdfPage page = docEvent.getPage();
	        String txt1 = "";
			String idioma = GuiaPDF.idioma;
			PdfFont font = null;
	        
			if (ConstantesGuiaDocente.IDIOMA_CASTELLANO.equals(idioma)) {
				txt1 = GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_C();
				
			} else if (ConstantesGuiaDocente.IDIOMA_VALENCIANO.equals(idioma)) {
				txt1 = GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_V();
				
			} else if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma)) {
				txt1 = GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_I();
				
			} else {
				txt1 = GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_C();
				
			}
			Integer pageNum = docEvent.getDocument().getPageNumber(page);
			font = PdfFontFactory.createFont(StandardFonts.HELVETICA_OBLIQUE);
			tablaPie = new Table(new float[]{92, 8}, true); 
			tablaPie.setWidth(UnitValue.createPercentValue(100));
	        
	        Paragraph p = new Paragraph();
			p.add(txt1);
			Cell c = new Cell();
			c.add(p);
			c.setBorder(Border.NO_BORDER);
		    c.setTextAlignment(TextAlignment.LEFT);
		    c.setPaddingTop(10);
			tablaPie.setFont(font);
			tablaPie.setFontSize(9.0f);
			tablaPie.addCell(c);
			
			Paragraph p2 = new Paragraph();
			p2.add(pageNum.toString());
			Cell c2 = new Cell();
			c2.add(p2);
			c2.setBorder(Border.NO_BORDER);
			c2.setTextAlignment(TextAlignment.CENTER);
			c2.setPaddingTop(10);
			tablaPie.addCell(c2);            
			
		} catch (IOException e) {
			LOGGER.info("Error IOException al crear la tabla de pie: " + e.getMessage());
		}  
		return tablaPie;
    }	
    

    /**
     * Manejador del evento de cambio de pagina, agrega el encabezado y pie de pagina
     * @param event Evento de pagina
     */
	@Override
    public void handleEvent(Event event) {
    	String txt1 = "";
    	String idioma = GuiaPDF.idioma;
        PdfDocumentEvent docEvent = (PdfDocumentEvent) event;
        PdfDocument pdfDoc = docEvent.getDocument();
        PdfPage page = docEvent.getPage();
        PdfCanvas canvas = new PdfCanvas(page.newContentStreamBefore(), page.getResources(), pdfDoc);        
        
        switch(idioma) {
	        case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
				txt1 = GuiaConstantes.TEXT_GUIA_DOCENTE_CAB_VALE + " \n" + GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_V();
				break;
	        case ConstantesGuiaDocente.IDIOMA_INGLES:
				txt1 = GuiaConstantes.TEXT_GUIA_DOCENTE_CAB_INGL + " \n" + GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_I();
				break;
	        default:
				txt1 = GuiaConstantes.TEXT_GUIA_DOCENTE_CAB_CAST + " \n" + GuiaPDF.gd.getModulo() + " " + GuiaPDF.gd.getNombre_C();
				break;
        }
        
        Table tablaEncabezado = this.crearTablaEncabezado(txt1);
        Rectangle rectanguloEncabezado = this.crearRectanguloEncabezado(docEvent);
        Canvas canvasEncabezado = new Canvas(canvas, rectanguloEncabezado);
        canvasEncabezado.add(tablaEncabezado);
        canvasEncabezado.close();

        Table tablaNumeracion = this.crearTablaPie(docEvent);
        Rectangle rectanguloPie = this.crearRectanguloPie(docEvent);
        Canvas canvasPie = new Canvas(canvas, rectanguloPie);
        canvasPie.add(tablaNumeracion);
        canvasPie.close();
        
        
        insertarMarcaAgua(page, pdfDoc);
    
    } 
    
    @SuppressWarnings("resource")
	private void insertarMarcaAgua(PdfPage page, PdfDocument pdfDoc){
    	String idioma = GuiaPDF.idioma; 
    	String marcaAgua = "";
    	PdfFont font = null;
    	
        try {
        	switch (idioma){
				case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
 					marcaAgua = GuiaConstantes.TEXT_GUIA_DOCENTE_BOR_VALE;
					break;
				case ConstantesGuiaDocente.IDIOMA_INGLES:
					marcaAgua = GuiaConstantes.TEXT_GUIA_DOCENTE_BOR_INGL;
					break;
				default:
					marcaAgua = GuiaConstantes.TEXT_GUIA_DOCENTE_BOR_CAST;
					break;				
			}        	
        	font = PdfFontFactory.createFont(StandardFonts.HELVETICA_BOLD);
            
        	PdfCanvas canvas = new PdfCanvas(page.newContentStreamBefore(), page.getResources(), pdfDoc);            
            
            if (!"F".equals(GuiaPDF.gd.getEstado())){
            	new Canvas(canvas, page.getPageSize())
                .setFontColor(ColorConstants.LIGHT_GRAY)
                .setFontSize(88)
                .setFont(font)
                .showTextAligned(new Paragraph(marcaAgua), 297.5f, 421, pdfDoc.getPageNumber(page),
                        TextAlignment.CENTER, VerticalAlignment.MIDDLE, 45)
                .close();
			} else {
				Image image = new Image(ImageDataFactory.create(GuiaPDF.sResourcesPath + GuiaConstantes.FONDO));				
				image.setFixedPosition(40, 170);
				new Canvas(canvas, page.getPageSize())
				.add(image)
				.close();
			}
            
        } catch (IOException e) {
        	LOGGER.info("Error al insertar la marca de agua: " + e.getMessage());
        }        
    }
}