package es.uv.gui.pdf;

import java.io.IOException;
import java.text.DecimalFormat;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.TreeMap;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.lang.StringEscapeUtils;

import com.itextpdf.html2pdf.HtmlConverter;
import com.itextpdf.io.font.constants.StandardFonts;
import com.itextpdf.kernel.colors.Color;
import com.itextpdf.kernel.colors.ColorConstants;
import com.itextpdf.kernel.colors.DeviceRgb;
import com.itextpdf.kernel.events.PdfDocumentEvent;
import com.itextpdf.kernel.font.PdfFont;
import com.itextpdf.kernel.font.PdfFontFactory;
import com.itextpdf.kernel.geom.PageSize;
import com.itextpdf.kernel.geom.Rectangle;
import com.itextpdf.kernel.pdf.PdfDocument;
import com.itextpdf.kernel.pdf.PdfWriter;
import com.itextpdf.layout.Document;
import com.itextpdf.layout.borders.Border;
import com.itextpdf.layout.borders.SolidBorder;
import com.itextpdf.layout.element.AreaBreak;
import com.itextpdf.layout.element.Cell;
import com.itextpdf.layout.element.IBlockElement;
import com.itextpdf.layout.element.IElement;
import com.itextpdf.layout.element.Paragraph;
import com.itextpdf.layout.element.Table;
import com.itextpdf.layout.layout.LayoutArea;
import com.itextpdf.layout.properties.AreaBreakType;
import com.itextpdf.layout.properties.HorizontalAlignment;
import com.itextpdf.layout.properties.OverflowWrapPropertyValue;
import com.itextpdf.layout.properties.Property;
import com.itextpdf.layout.properties.TextAlignment;
import com.itextpdf.layout.properties.UnitValue;

import com.uv.common.ejb.i18n.InternationalizedFactory;
import com.uv.common.ejb.i18n.language.LanguageFoo;
import com.uv.common.filters.utiles.FuncionesUtiles;
import com.uv.common.util.ByteArrayOutputStreamUtil;

import es.uv.guia.utiles.ConstantesGuiaDocente;

public class GuiaPDF {
	
	private static final Logger LOGGER = Logger.getLogger("es.uv.gui.pdf.GuiaPDF");
	
	public static String idioma            = ConstantesGuiaDocente.IDIOMA_CASTELLANO;	
	private Document document              = null;
	private ByteArrayOutputStreamUtil baos = null;
	
	private Color colorCabecera;
	private Color colorCabeceraSub;
	private Color colorCabeceraSeccion;
	private Color colorFondoTablaCabecera;
	private String  fontCabeceraSeccion = "";
	private PdfFont fontEtiquetaTabla   = null;
	private PdfFont fontContenidoTabla  = null;
	
	public static String sResourcesPath;
	public static Guia gd;
	private HashMap<String, Object> mapa;
    
	// Textos del PDF:		
	public GuiaPDF (Guia guiad, HashMap<String, Object> mapaOriginal, String sResourcesPath) {
		GuiaPDF.gd = guiad;	
		this.mapa = mapaOriginal;
		GuiaPDF.sResourcesPath = sResourcesPath;
		GuiaPDF.idioma = (String)mapa.get(GuiaConstantes.PDF_IDIOMA);
		
		LanguageFoo langFooAplica = (LanguageFoo)mapa.get(ConstantesGuiaDocente.TEXT_IDIOMA_ACTUAL_ID);		
		LanguageFoo langPDF = new LanguageFoo();
		
		
		int langId = ConstantesGuiaDocente.IDIOMA_CASTELLANO_I;	
		
		switch(idioma) {
			case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
				langId = ConstantesGuiaDocente.IDIOMA_VALENCIANO_I;
				langPDF.setId(langId);
				langPDF.setMainFrameCode(ConstantesGuiaDocente.IDIOMA_VALENCIANO_C);
				langPDF.setLocale(new Locale("ca","ES"));
				break;
			case ConstantesGuiaDocente.IDIOMA_INGLES:
				langId = ConstantesGuiaDocente.IDIOMA_INGLES_I;
				langPDF.setId(langId);
				langPDF.setMainFrameCode(ConstantesGuiaDocente.IDIOMA_VALENCIANO_C);
				langPDF.setLocale(new Locale("en","ES"));
				break;
			default:
				langId = ConstantesGuiaDocente.IDIOMA_CASTELLANO_I;
				langPDF.setId(langId);
				langPDF.setMainFrameCode(ConstantesGuiaDocente.IDIOMA_CASTELLANO_C);
				langPDF.setLocale(new Locale("es","ES"));
		}
		
		this.mapa.put(ConstantesGuiaDocente.TEXT_IDIOMA_ACTUAL_ID, langPDF);
		this.createPDF();
		
		// SE RESTAURA EL IDIOMA DE LA APLICACION
		mapa.put(ConstantesGuiaDocente.TEXT_IDIOMA_ACTUAL_ID, langFooAplica);	
		
	}
	
	private void createPDF () {			
		baos = new ByteArrayOutputStreamUtil();
		try {	
			
			initFonts ();
			
			initColors();
			
			//Inicializar mapeos
			initMapeos();			

			// Inicializar Documento PDF
			PdfWriter pdfWriter = new PdfWriter(baos);
			PdfDocument pdfDocument = new PdfDocument(pdfWriter);
			document = new Document(pdfDocument, PageSize.A4); 
			
            GuiaPDFEvents evento = new GuiaPDFEvents(document);
            pdfDocument.addEventHandler(PdfDocumentEvent.END_PAGE, evento);
            
            document.setMargins(110, 40, 40, 40);
            
            insertLineaBlanco ();
            insertLineaBlanco ();
            
            //Insertar Apartado 1: FICHA IDENTIFICATIVA
			insertFichaIdentificativa();		
		    
		    //Insertar Apartado 2: RESUMEN
		    insertResumen();
		    
		    //Insertar Apartado 3: CONOCIMIENTOS PREVIOS
			insertConocimientosPrevios();
			
			//Insertar Apartado 4: COMPETENCIAS
			insertCompetencias();
			
			//Insertar Apartado 5: RESULTADOS DEL APRENDIZAJE
			insertResultadosAprendizaje();
			
			//Insertar Apartado 6: DESCRIPCION DE CONTENIDOS
			insertDescripcionContenidos();
			
			//Insertar Apartado 7: VOLUMEN DE TRABAJO
			insertVolumenTrabajo();
			
			//Insertar Apartado 8: METODOLOGIA DOCENTE 
			insertMetodologia();
			
			//Insertar Apartado 9: EVALUACION			
			insertEvaluacion();
		    
			//Insertar Apartado 10: REFERENCIAS			
			insertReferencias();
			
			if(FuncionesUtiles.mostrarApartado11(gd.getCursosAcadPermitidosApartado11(), (String) this.mapa.get(ConstantesGuiaDocente.TEXT_CURSO_ACAD_U))){
				//Insertar Apartado 11: ADENDA
				insertAdenda();	
			}
		    
		    document.close();
 
            pdfWriter.close();
            
            LOGGER.info("PDF creado");				
			
		}catch (Exception e) {
			LOGGER.info("Error al crear el PDF: " + e.getMessage());
		}			
	}
	
	/**
	 * initMapeos: Inicializa los mapeos para optimizar el documento
	 *
	 */
	private void initMapeos () {		
		Utils.initMapeos();		
	}
	
	private void initFonts () {	
		this.fontCabeceraSeccion = StandardFonts.HELVETICA_BOLD;
		try {
			this.fontEtiquetaTabla  = PdfFontFactory.createFont(StandardFonts.HELVETICA_BOLD);
			this.fontContenidoTabla = PdfFontFactory.createFont(StandardFonts.HELVETICA);
			
		} catch (IOException e) {
			LOGGER.info("Error inicializando fuentes: " + e.getMessage());
		}
	}
	
	private void initColors () {		
		this.colorCabecera           = new DeviceRgb(1, 57, 82); //AZUL
		this.colorCabeceraSub        = new DeviceRgb(203, 192, 160); //BEIGE
		this.colorCabeceraSeccion    = ColorConstants.WHITE;
		this.colorFondoTablaCabecera = new DeviceRgb(230, 227, 210);
	}
	
	/**
	 * 
	 * @return: El BytaArrayOutputStream del PDF.
	 */
	public ByteArrayOutputStreamUtil getPDFByteArrayOutputStream () {
		return baos;
	}
	
	
	/**    
     * Se inserta la seccion Ficha identificativa
     * */
	private void insertFichaIdentificativa() { 		
		
		//cabecera seccion FICHA IDENTIFICATIVA
		insertCabecera (traduce(GuiaConstantes.TEXT_1_FICHA_IDENTIFICATIVA), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false );
		
		insertLineaBlanco();		
		
		//cabecera subseccion Datos de la Asignatura
		insertCabecera (traduce(GuiaConstantes.TEXT_1_1_DATOS_ASIGNATURA), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false );
			
		//tabla Datos de la Asignatura
		insertTablaDatosAsignatura();	
		
		insertLineaBlanco();			
		
		//insertamos contenido de la subseccion Titulaciones
		insertSubSeccionTitulaciones();	
		
		insertLineaBlanco();
		
		//insertamos contenido de la subseccion Materias
		insertSubSeccionMaterias();
		
		insertLineaBlanco();
				
		//insertamos contenido de la subseccion Coordinacion
		insertSubSeccionCoordinacion();
		
		insertLineaBlanco();    
	}

	
	/**
     * Se inserta la tabla de datos de la asignatura
     * */
	private void insertTablaDatosAsignatura () {
		
		try {
			Table tabla = new Table(2);
			tabla.setWidth(UnitValue.createPercentValue(100));
			
			contenidoTablaFichIdent(tabla, traduce(GuiaConstantes.TEXT_CODIGO), ""+gd.getModulo());
			String nombre = gd.getNombre_C();
			
			if (ConstantesGuiaDocente.IDIOMA_VALENCIANO.equalsIgnoreCase(idioma) ) {
				nombre = gd.getNombre_V();
			} else if (ConstantesGuiaDocente.IDIOMA_INGLES.equalsIgnoreCase(idioma) ) {
				nombre = gd.getNombre_I();
			}
			contenidoTablaFichIdent(tabla, traduce(GuiaConstantes.TEXT_NOMBRE), nombre );
			contenidoTablaFichIdent(tabla, traduce(GuiaConstantes.TEXT_CICLO), Utils.getValorMapeo(ConstantesGuiaDocente.TEXT_TIPO_TITULACION, GuiaPDF.idioma, gd.getCiclo()));
			contenidoTablaFichIdent(tabla, traduce(GuiaConstantes.TEXT_C_ECTS), String.valueOf(Double.parseDouble(gd.getCredit())));
			String cursoAcad = Integer.parseInt((gd.getCursoAcad())) - 1 + " - "  + gd.getCursoAcad();
			contenidoTablaFichIdent(tabla, traduce(GuiaConstantes.TEXT_CURSO_ACAD), cursoAcad );	
			
			document.add(tabla);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar la tabla de datos de la asignatura: " + e.getMessage());
		}
		
	}
	
	/**
     * Se inserta el contenido de la tabla de la seccion ficha identificativa
     * */
	private void contenidoTablaFichIdent(Table tabla, String etiqueta, String contenido ){
		Cell cell1 = null;
		Cell cell2 = null;	
		try{
			Paragraph p1 = new Paragraph();
			p1.add(etiqueta);
			p1.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			p1.setFixedLeading(GuiaConstantes.INTERLINEADO);
			cell1 = new Cell ();
			cell1.add(p1);
			cell1.setWidth(110);			
			cell1.setFontSize(11);
			cell1.setFont(PdfFontFactory.createFont(StandardFonts.HELVETICA_BOLD));
			
			Paragraph p2 = new Paragraph();
			p2.add(contenido);
			p2.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			p2.setFixedLeading(GuiaConstantes.INTERLINEADO);
			cell2 = new Cell ();
			cell2.add(p2);			
			cell2.setFontSize(11);
			cell2.setFont(PdfFontFactory.createFont(StandardFonts.HELVETICA));
			
			tabla.addCell(cell1);
			tabla.addCell(cell2);		
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar el contenido de la tabla de la ficha identificativa: " + e.getMessage());
		}	
	}
	
	/**
     * Se inserta las etiquetas de la subseccion titulacion
     * */
	private void insertEtiquetasCabeceraTitulacion () {				
		
		try {
			Table tabla = new Table(new float[]{42, 35, 7, 15}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));
						
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_TITULACION);
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_CENTRO);
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_CURSO);
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_PERIODO);			
			
			document.add(tabla);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar las etiquetas de la cabecera Titulacion: " + e.getMessage());
		}
		
	}	
	
	/**
     * Se inserta la subseccion titulaciones
     * */
	private void insertSubSeccionTitulaciones () {
		String titulacion = "";
		String centro 	  = "";
		String curso 	  = "";
		String periodo 	  = "";
		Table tabla = null;
		
		
		TreeMap<?, ?> tmTitulaciones = gd.getTmTitulaciones();		
		
		if ( !tmTitulaciones.isEmpty()) {
			
			//cabecera subseccion Titulaciones
			insertCabecera (traduce(GuiaConstantes.TEXT_1_2_TITULACIONES), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false );
			
			Iterator<?> itTitulaciones = tmTitulaciones.keySet().iterator();	
			
			tabla = new Table(new float[]{42, 35, 7, 15}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));
			
			//etiquetas de la subseccion Titulaciones
			insertEtiquetasCabeceraTitulacion();
			
			while ( itTitulaciones.hasNext() ) {
				TitulacionGuia titGuia = (TitulacionGuia)tmTitulaciones.get(itTitulaciones.next());

				switch (idioma){
					case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
							titulacion = titGuia.getTitulacion() + " - " + titGuia.getNombre_C();
							centro = titGuia.getCentro_C();
							curso = titGuia.getCurso();
							periodo = Utils.getValorMapeo(GuiaConstantes.PERIODO, GuiaPDF.idioma, titGuia.getPeriodo());
						break;
					case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
							titulacion = titGuia.getTitulacion() + " - " + titGuia.getNombre_V();
							centro = titGuia.getCentro_V();
							curso = titGuia.getCurso();
							periodo = Utils.getValorMapeo(GuiaConstantes.PERIODO, GuiaPDF.idioma, titGuia.getPeriodo());
						break;
					case ConstantesGuiaDocente.IDIOMA_INGLES:
							titulacion = titGuia.getTitulacion() + " - " + titGuia.getNombre_I();
							centro = titGuia.getCentro_I();
							curso = titGuia.getCurso();
							periodo = Utils.getValorMapeo(GuiaConstantes.PERIODO, GuiaPDF.idioma, titGuia.getPeriodo());
						break;
					default:
							String nombre = titGuia.getNombre_C();
							titulacion = titGuia.getTitulacion() + " - " + nombre;
							centro = titGuia.getCentro_C();
							curso = titGuia.getCurso();
							periodo = Utils.getValorMapeo(GuiaConstantes.PERIODO, GuiaPDF.idioma, titGuia.getPeriodo());
						break;				
				}
				
				if(GuiaConstantes.PERIODO_NULO.equals(periodo)){	
					
					switch (idioma){
						case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
							 periodo = GuiaConstantes.TEXT_PERIODO_NULO_CAST;
							 break;
						case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
							 periodo = GuiaConstantes.TEXT_PERIODO_NULO_VALE;
							 break;
						case ConstantesGuiaDocente.IDIOMA_INGLES:
							 periodo = GuiaConstantes.TEXT_PERIODO_NULO_INGL;
							 break;
						default:
							 periodo = GuiaConstantes.TEXT_PERIODO_NULO_CAST;
							 break;				
					}
				}
				
				insertContentSubSeccion(tabla, titulacion, TextAlignment.LEFT);	
				insertContentSubSeccion(tabla, centro, TextAlignment.LEFT);
				insertContentSubSeccion(tabla, curso, TextAlignment.CENTER);
				insertContentSubSeccion(tabla, periodo, TextAlignment.LEFT);
				
			}
			document.add(tabla);
		}
		
	}	
	
	/**
     * Se inserta la subseccion Materias
     * */
	private void insertSubSeccionMaterias(){
		
		String titulacion = "";
		String materia 	  = "";
		String caracter   = "";
		
		ArrayList<?> alMaterias = gd.getAlMaterias();
		
		if ( !alMaterias.isEmpty()) {	
			
			//cabecera subseccion Materias
			insertCabecera (traduce(GuiaConstantes.TEXT_1_3_MATERIAS), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false );	
			
			//etiquetas de la subseccion Titulaciones
			insertEtiquetasCabeceraMaterias();
		    
			Table tabla = new Table(new float[]{42, 36, 22}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));
			
			for ( int i=0; i<alMaterias.size(); i++) {
				MateriaTitulacionGuia materiaGuia = (MateriaTitulacionGuia)alMaterias.get(i);
				
				switch (idioma){
					case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
							titulacion = Integer.toString(materiaGuia.getTitulacion()) + " - " + materiaGuia.getTitulacion_C();
							materia    = materiaGuia.getMateria() + " - " + materiaGuia.getNombre_C();
							caracter   = Utils.getValorMapeo(GuiaConstantes.TIPO, GuiaPDF.idioma, materiaGuia.getCaracter());
						break;
					case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
							titulacion = Integer.toString(materiaGuia.getTitulacion()) + " - " + materiaGuia.getTitulacion_V();
							materia    = materiaGuia.getMateria() + " - " + materiaGuia.getNombre_V();
							caracter   = Utils.getValorMapeo(GuiaConstantes.TIPO, GuiaPDF.idioma, materiaGuia.getCaracter());
						break;
					case ConstantesGuiaDocente.IDIOMA_INGLES:
							titulacion = Integer.toString(materiaGuia.getTitulacion()) + " - " + materiaGuia.getTitulacion_I();
							materia    = materiaGuia.getMateria() + " - " + materiaGuia.getNombre_I();
							caracter   = Utils.getValorMapeo(GuiaConstantes.TIPO, GuiaPDF.idioma, materiaGuia.getCaracter());
						break;
					default:
						    String titulacionC = materiaGuia.getTitulacion_C();
							titulacion = Integer.toString(materiaGuia.getTitulacion()) + " - " + titulacionC;
							materia    = materiaGuia.getMateria() + " - " + materiaGuia.getNombre_C();
							caracter   = Utils.getValorMapeo(GuiaConstantes.TIPO, GuiaPDF.idioma, materiaGuia.getCaracter());
						break;				
				}
				
				insertContentSubSeccion(tabla, titulacion, TextAlignment.LEFT);	
				insertContentSubSeccion(tabla, materia, TextAlignment.LEFT);
				insertContentSubSeccion(tabla, caracter, TextAlignment.LEFT);			
			}
			document.add(tabla);
		}
		
	}	
	
	/**
     * Se insertan las etiquetas de la subseccion Materias
     * */
	private void insertEtiquetasCabeceraMaterias () {				
		
		try {
			Table tabla = new Table(new float[]{42, 36, 22}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));
						
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_TITULACION);
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_MATERIA);
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_CARACTER);		
			
			document.add(tabla);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar las etiquetas de la cabecera Materias: " + e.getMessage());
		}
		
	}
	
	/**
     * Se inserta la subseccion Coordinacion
     * */
	private void insertSubSeccionCoordinacion(){
		String nombre       = "";
		String departamento = "";
		
		ArrayList <?> alCoordinadores = gd.getAlCoordinadores();
		
		if ( !alCoordinadores.isEmpty()) {
			//cabecera subseccion Coordinacion
			insertCabecera (traduce(GuiaConstantes.TEXT_1_4_COORDINACION), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);	
			
			//etiquetas de la subseccion Titulaciones
			insertEtiquetasCabeceraCoordinacion();
		    
			Table tabla = new Table(new float[]{50, 50}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));		
		
			for ( int i=0; i<alCoordinadores.size(); i++) {
				CoordinadorGuia coorGuia = (CoordinadorGuia)alCoordinadores.get(i);
	
				switch (idioma){
					case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
							nombre       = (coorGuia.getApellidos()).trim() + ", " + coorGuia.getNombre();
							departamento = coorGuia.getDepartamento() + " - " + coorGuia.getDepartamento_C();
						break;
					case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
							nombre       = (coorGuia.getApellidos()).trim() + ", " + coorGuia.getNombre();
							departamento = coorGuia.getDepartamento() + " - " + coorGuia.getDepartamento_V();
						break;
					case ConstantesGuiaDocente.IDIOMA_INGLES:
							nombre       = (coorGuia.getApellidos()).trim() + ", " + coorGuia.getNombre();
							departamento = coorGuia.getDepartamento() + " - " + coorGuia.getDepartamento_I();
						break;
					default:
						    String departamentoC = coorGuia.getDepartamento_C();
							nombre       = (coorGuia.getApellidos()).trim() + ", " + coorGuia.getNombre();
							departamento = coorGuia.getDepartamento() + " - " + departamentoC;
						break;				
				}
				
				insertContentSubSeccion(tabla, nombre, TextAlignment.LEFT);	
				insertContentSubSeccion(tabla, departamento, TextAlignment.LEFT);				
			}
			document.add(tabla);
		}
	}
	
	/**
     * Se insertan las etiquetas de la subseccion Coordinacion
     * */
	private void insertEtiquetasCabeceraCoordinacion () {				
		
		try {
			Table tabla = new Table(new float[]{50, 50}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));
						
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_NOMDEPARTAMENTO);
			etiquetaSubSeccion(tabla, GuiaConstantes.TEXT_DEPARTAMENTO);		
			
			document.add(tabla);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar las etiquetas de la cabecera Coordinacion: " + e.getMessage());
		}
		
	}
	
	/**
     * Se inserta la seccion Resumen
     * */
	private void insertResumen () {	
		
		try {
			
			insertLineaBlanco();
			
			//cabecera seccion RESUMEN
		    insertCabecera (traduce(GuiaConstantes.TEXT_2_RESUMEN), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false );
		    					
			String htmlText = gd.getSbResumen().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
			
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && "".equals(htmlText)){
			    insertCabecera (GuiaConstantes.ENGLISH_VERSION, null, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false );
				insertApartadoBlob (gd.getSbResumenMaestro().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL));
			} else {
				insertApartadoBlob (htmlText);
			}			
				
		} catch (Exception e){
			LOGGER.info("Error insertando Resumen: " + e.getMessage());
		}	
		
	}	
	
	/**
     * Se inserta la seccion Conocimientos Previos
     * */
	private void insertConocimientosPrevios () {
			
		insertLineaBlanco();
		
		//cabecera seccion CONOCIMIENTOS PREVIOS
	    insertCabecera (traduce(GuiaConstantes.TEXT_3_CONOCIMIENTOS), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);
	    
	    insertSubSeccionRelacion();	
	    
	    insertSubSeccionOtrosTipos();	
	}
	
	/**
     * Se inserta la subseccion Relacion
     * */
	private void insertSubSeccionRelacion (){
		String estudioTitulacion = "";
		String idUnidadNombre    = "";		
		String tipoRes          = "";
		int estudioCabecera      = 0;
		
		try {						
			
			ArrayList <?> alConocimientos = gd.getAlConocimientos();
			Table tabla = new Table(new float[]{100}, true);
			tabla.setWidth(UnitValue.createPercentValue(100));
			
			if ( !alConocimientos.isEmpty()) {
			  
				insertLineaBlanco();
				
				//cabecera subseccion
				insertCabecera (traduce(GuiaConstantes.TEXT_3_1_RELACION), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false );		
				
				for ( int i=0; i<alConocimientos.size(); i++) {
					ConocimientoGuia conocimientoGuia = (ConocimientoGuia)alConocimientos.get(i);
					
					int estudio = conocimientoGuia.getEstudio();
					if (estudioCabecera != estudio){
						estudioCabecera = estudio;
						
						estudioTitulacion = getEstudioTitulacion(estudio, conocimientoGuia);
						
						//cabecera subseccion
						insertCabecera (estudioTitulacion, this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false );												
					}
					
					String res = (conocimientoGuia.getResequ()).trim();
					int tipo = conocimientoGuia.getTipo();
					if (!tipoRes.equalsIgnoreCase(tipo+res)){
						tipoRes = tipo+res;						
						
						insertContentSubSeccion(tabla, Utils.getValorMapeo(GuiaConstantes.TEXT_RESTRICCION, GuiaPDF.idioma, Integer.toString(conocimientoGuia.getTipo())), TextAlignment.LEFT);			
					}
					
					switch (idioma){
						case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
								idUnidadNombre = GuiaConstantes.TABULACION + conocimientoGuia.getIdUnidad() + " - " + conocimientoGuia.getNombre_C();
							break;
						case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
								idUnidadNombre = GuiaConstantes.TABULACION + conocimientoGuia.getIdUnidad() + " - " + conocimientoGuia.getNombre_V();
							break;
						case ConstantesGuiaDocente.IDIOMA_INGLES:
								idUnidadNombre = GuiaConstantes.TABULACION + conocimientoGuia.getIdUnidad() + " - " + conocimientoGuia.getNombre_I();
							break;
						default:
								idUnidadNombre = GuiaConstantes.TABULACION + conocimientoGuia.getIdUnidad() + " - " + conocimientoGuia.getNombre_C();
							break;				
					}
					
					insertContentSubSeccion(tabla, idUnidadNombre, TextAlignment.LEFT);						
				}
				document.add(tabla);
			} else {				
				insertLineaBlanco();
				
				//cabecera subseccion
				insertCabecera (traduce(GuiaConstantes.TEXT_3_1_RELACION), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
				
				insertContentSubSeccion(tabla, traduce(GuiaConstantes.ALERT_RESTRICCION), TextAlignment.LEFT);
				document.add(tabla);
			}			
					
		} catch (Exception e) {
			LOGGER.info("Error al insertar la subseccion Relacion: " + e.getMessage());
		}			
		
	}
	
	/**
     * Devuelve el Estudio-Titulacion en el idioma correcto
     * */
	private String getEstudioTitulacion(int estudio, ConocimientoGuia conocimientoGuia){
		String estudioTitulacion = "";
		switch (idioma){
			case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
					estudioTitulacion = estudio + " - " + conocimientoGuia.getTitulacion_C().trim() + " :";
				break;
			case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
					estudioTitulacion = estudio + " - " + conocimientoGuia.getTitulacion_V().trim() + " :";
				break;
			case ConstantesGuiaDocente.IDIOMA_INGLES:
					estudioTitulacion = estudio + " - " + conocimientoGuia.getTitulacion_I().trim() + " :";
				break;
			default:
					estudioTitulacion = estudio + " - " + conocimientoGuia.getTitulacion_C().trim() + " :";
				break;				
		}
		return estudioTitulacion;
	}
	
	/**
     * Se inserta la subseccion Otros Tipos
     * */
	private void insertSubSeccionOtrosTipos(){
		Table tabla = new Table(1);
		tabla.setWidth(UnitValue.createPercentValue(100));
		
		switch (idioma){
			case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
					if (!"".equals(gd.getOtrosRequisitos_C())){
						insertLineaBlanco();
						//cabecera subseccion
						insertCabecera (traduce(GuiaConstantes.TEXT_3_2_OTRO_TIPO), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
						insertContentSubSeccion(tabla, gd.getOtrosRequisitos_C(), TextAlignment.JUSTIFIED);
					}
				break;
			case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
					if (!"".equals(gd.getOtrosRequisitos_V())){
						insertLineaBlanco();
						//cabecera subseccion
						insertCabecera (traduce(GuiaConstantes.TEXT_3_2_OTRO_TIPO), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
						insertContentSubSeccion(tabla, gd.getOtrosRequisitos_V(), TextAlignment.JUSTIFIED);
					}
				break;
			case ConstantesGuiaDocente.IDIOMA_INGLES:
					if (!"".equals(gd.getOtrosRequisitos_I())){
						insertLineaBlanco();
						//cabecera subseccion
						insertCabecera (traduce(GuiaConstantes.TEXT_3_2_OTRO_TIPO), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
						insertContentSubSeccion(tabla, gd.getOtrosRequisitos_I(), TextAlignment.JUSTIFIED);
					}
				break;
			default:
					if (!"".equals(gd.getOtrosRequisitos_C())){
						insertLineaBlanco();
						//cabecera subseccion
						insertCabecera (traduce(GuiaConstantes.TEXT_3_2_OTRO_TIPO), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
						insertContentSubSeccion(tabla, gd.getOtrosRequisitos_C(), TextAlignment.JUSTIFIED);
					}
				break;				
		}
		document.add(tabla);
	}
	
	/**
     * Se inserta la seccion Competencias
     * */
	private void insertCompetencias(){
		String estudioTitulacion = "";
		String nombre            = "";
		
		try {
					
			ArrayList<?> alCompetencias = gd.getAlCompetencias();
						
			insertLineaBlanco();
			
			//cabecera seccion COMPETENCIAS
		    insertCabecera (traduce(GuiaConstantes.TEXT_4_COMPETENCIAS), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);				
					
			int estudioComp = 0;
			for ( int i=0; i<alCompetencias.size(); i++) {
				Table tabla = new Table(new float[]{3, 97}, true);
				tabla.setWidth(UnitValue.createPercentValue(100));
				Competencia comp = (Competencia)alCompetencias.get(i);				
				
				if (comp.getEstudio() != estudioComp) {
					insertLineaBlanco();
					switch (idioma){
						case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
								estudioTitulacion = comp.getEstudio() + " - " + comp.getTitulacion_C();
							break;
						case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
								estudioTitulacion = comp.getEstudio() + " - " + comp.getTitulacion_V();
							break;
						case ConstantesGuiaDocente.IDIOMA_INGLES:
								estudioTitulacion = comp.getEstudio() + " - " + comp.getTitulacion_I();
							break;
						default:
								estudioTitulacion = comp.getEstudio() + " - " + comp.getTitulacion_C();
							break;				
					}
					//cabecera subseccion
					insertCabecera (estudioTitulacion, this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
				}
				
				switch (idioma){
					case ConstantesGuiaDocente.IDIOMA_CASTELLANO:
							nombre = comp.getNombre_C().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
						break;
					case ConstantesGuiaDocente.IDIOMA_VALENCIANO:
							nombre = comp.getNombre_V().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
						break;
					case ConstantesGuiaDocente.IDIOMA_INGLES:
							nombre = comp.getNombre_I().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
						break;
					default:
							nombre = comp.getNombre_C().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
						break;				
				}
				
				estudioComp = comp.getEstudio();
				
				insertContentSubSeccionLista(tabla, nombre, TextAlignment.JUSTIFIED);

				document.add(tabla);
			}
			
			insertLineaColor(this.colorCabeceraSub);
				
		} catch (Exception e){
			LOGGER.info("Error insertando Competencias: " + e.getMessage());
		}	
	}
	
	/**
     * Se inserta la seccion Resultados del Aprendizaje 
     * */
	private void insertResultadosAprendizaje () {	
		
		try {
			
			insertLineaBlanco();
			
			//cabecera seccion RESULTADOS DEL APRENDIZAJE
		    insertCabecera (traduce(GuiaConstantes.TEXT_5_RESULTADOS_APRENDIZAJE), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);
		    			
			// GUIADOCENTE-31. Error del apstrofe.
			String htmlText = gd.getSbResultadosAprendizaje().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
			
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && "".equals(htmlText)){
				//cabecera English version is not available
			    insertCabecera (GuiaConstantes.ENGLISH_VERSION, this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false );
			  
				// SE MUESTRA EL APARTADO EN IDIOMA INGLES PERO CON ADVERTENCIA		
				// GUIADOCENTE-31. Error del apstrofe.
				insertApartadoBlob (gd.getSbResultadosAprendizaje().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL));
				
			} else {
				
				insertApartadoBlob (htmlText);
								
			}			
				
		} catch (Exception e){
			LOGGER.info("Error insertando Resultados del Aprendizaje: " + e.getMessage());
		}	
		
	}
	
	/**
     * Se inserta la seccion Descripcion de Contenidos
     * */
	private void insertDescripcionContenidos () {	
		String tema      = "";
		String contenido = "";
		
		try {				
			
			TreeMap<?, ?> tmUnidades= gd.getTmUnidades();
			
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && (tmUnidades.isEmpty() || !gd.getUnidadesIngles(tmUnidades))){
				// SE OMITE EL APARTADO EN IDIOMA INGLES
			} else {
				
				insertLineaBlanco();
				
				//cabecera seccion DESCRIPCION DE CONTENIDOS
			    insertCabecera (traduce(GuiaConstantes.TEXT_6_CONTENIDOS), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);			
				
				Iterator<?> itUnidades = tmUnidades.keySet().iterator();
				
				while ( itUnidades.hasNext() ) {
					Table tabla = new Table(1);
					tabla.setWidth(UnitValue.createPercentValue(100));
					
					UnidadGuia unidGuia = (UnidadGuia)tmUnidades.get(itUnidades.next());
					insertLineaBlanco();
					//cabecera subseccion
					tema = unidGuia.getNum_Tema() + ". " + unidGuia.getNombre().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
					insertCabecera (tema, this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
					
					contenido = unidGuia.getDescripcion().trim().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
					insertContentSubSeccion(tabla, contenido, TextAlignment.JUSTIFIED);			
					
					document.add(tabla);
					
					insertLineaColor(this.colorCabeceraSub);
				}				
			}			
				
		} catch (Exception e){
			LOGGER.info("Error insertando Descripcion de Contenidos: " + e.getMessage());
		}	
		
	}
	
	/**
     * Se inserta la seccion Volumen de Trabajo
     * */
	private void insertVolumenTrabajo () {
		
		String etiqActividad       = "";
		String etiqHoras           = "";
		String etiqPresencial      = "";
		String contenidoActividad  = "";
		String contenidoHoras      = "";
		String contenidoPresencial = "";
		double total = 0.0;
		double totalPresenciales = 0.0;
		double totalNoPresenciales = 0.0;	
		DecimalFormat decimalFormat = new DecimalFormat("0.00");
		
		try {
			insertSaltoPagina(230);
			
			//cabecera seccion VOLUMEN DE TRABAJO
		    insertCabecera (traduce(GuiaConstantes.TEXT_7_VOLUMEN_TRABAJO), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);			
					
			insertLineaBlanco();
			
			//Creamos la tabla
			Table tabla = new Table(3);
			tabla.setWidth(UnitValue.createPercentValue(90));
			tabla.setHorizontalAlignment(HorizontalAlignment.CENTER);
			
			ArrayList<?> alTrabajoPresencial = gd.getAlTrabajoPresencial();
			ArrayList<?> alTrabajoNoPresencial = gd.getAlTrabajoNoPresencial();	
			
			if (!alTrabajoPresencial.isEmpty() || !alTrabajoNoPresencial.isEmpty()){
				etiqActividad  = (traduce(GuiaConstantes.TEXT_7_1_ACTIVIDADES)).toUpperCase();
				etiqHoras      = traduce(GuiaConstantes.TEXT_7_0_2_HORAS);
				etiqPresencial = "% " + traduce(GuiaConstantes.TEXT_7_0_3_POR_PRESENCIA);				
				
				contenidoTablaVolumenTrabajo(tabla, etiqActividad, etiqHoras, etiqPresencial, this.colorFondoTablaCabecera, fontEtiquetaTabla, false);					
			}
			
			if ( !alTrabajoPresencial.isEmpty()) {
				totalPresenciales = trabajoPresencial(alTrabajoPresencial, tabla);
			}
			
			if ( !alTrabajoNoPresencial.isEmpty()) {
				totalNoPresenciales = trabajoNoPresencial(alTrabajoNoPresencial, tabla);
			}
			
			total = totalPresenciales + totalNoPresenciales;
			
			if (!alTrabajoPresencial.isEmpty() || !alTrabajoNoPresencial.isEmpty()){
				contenidoActividad  = (traduce(GuiaConstantes.TEXT_7_0_1_TOTAL)).toUpperCase();
				contenidoHoras      = decimalFormat.format(total);
				contenidoPresencial = "";
				
				contenidoTablaVolumenTrabajo(tabla, contenidoActividad, contenidoHoras, contenidoPresencial, this.colorFondoTablaCabecera, fontEtiquetaTabla, true);
			}
			
			document.add(tabla);	
			
		} catch (Exception e){
			LOGGER.info("Error insertando Volumen de Trabajo: " + e.getMessage());
		}			
	}
	
	/**
     * Se calcula el trabajo presencial
     * */
	private double trabajoPresencial(ArrayList<?> alTrabajoPresencial, Table tabla){
		String contenidoActividad  = "";
		String contenidoHoras      = "";
		String contenidoPresencial = "";
		double totalPresenciales = 0.0;
		DecimalFormat decimalFormat = new DecimalFormat("0.00");
		String number = null;
		
		for ( int i=0; i<alTrabajoPresencial.size(); i++) {
			VolumenTrabajoPresencialGuia volPresencial = (VolumenTrabajoPresencialGuia)alTrabajoPresencial.get(i);
			String actividad = volPresencial.getActividad();
			int porPresencial = volPresencial.getPor_presencia();
			number = decimalFormat.format(Double.parseDouble(volPresencial.getHoras()));
			
			if (GuiaConstantes.PRACTICAS_CLINICAS.equals(actividad)){
				number = decimalFormat.format(3*Double.parseDouble(volPresencial.getHoras()));
			}
			if (GuiaConstantes.PRACTICAS_EXTERNAS.equals(actividad) || GuiaConstantes.TRABAJO_FIN_GRADO_MASTER.equals(actividad)){ 
				number = ""; 
			}
			
			contenidoActividad  = Utils.getValorMapeo(GuiaConstantes.TEXT_ACTIVIDAD, GuiaPDF.idioma, volPresencial.getActividad());
			contenidoHoras      = number;
			contenidoPresencial = "" + porPresencial;
			
			contenidoTablaVolumenTrabajo(tabla, contenidoActividad, contenidoHoras, contenidoPresencial, null, fontContenidoTabla, false);
			
			if (GuiaConstantes.PRACTICAS_EXTERNAS.equals(actividad) || GuiaConstantes.TRABAJO_FIN_GRADO_MASTER.equals(actividad)){ 
				totalPresenciales = totalPresenciales + 0; 
			} else {
				if (GuiaConstantes.PRACTICAS_CLINICAS.equals(actividad)){
					totalPresenciales = totalPresenciales + 3*Double.parseDouble(volPresencial.getHoras());
				} else {
					totalPresenciales = totalPresenciales + Double.parseDouble(volPresencial.getHoras());
				}
			}
		}
		return totalPresenciales;
	}
	
	/**
     * Se calcula el trabajo NO presencial
     * */
	private double trabajoNoPresencial(ArrayList<?> alTrabajoNoPresencial, Table tabla){
		String contenidoActividad  = "";
		String contenidoHoras      = "";
		String contenidoPresencial = "";
		double totalNoPresenciales = 0.0;	
		DecimalFormat decimalFormat = new DecimalFormat("0.00");
		String number = null;
		
		for ( int i=0; i<alTrabajoNoPresencial.size(); i++) {				  
			VolumenTrabajoNoPresencialGuia volNoPresencial = (VolumenTrabajoNoPresencialGuia)alTrabajoNoPresencial.get(i);

			int porPresencial = volNoPresencial.getPor_presencia();
			number = decimalFormat.format(Double.parseDouble(volNoPresencial.getHoras()));
			
			contenidoActividad  = volNoPresencial.getResumen().trim();
			contenidoHoras      = number;
			contenidoPresencial = "" + porPresencial;
			
			contenidoTablaVolumenTrabajo(tabla, contenidoActividad, contenidoHoras, contenidoPresencial, null, fontContenidoTabla, false);
			
			totalNoPresenciales = totalNoPresenciales + Double.parseDouble(volNoPresencial.getHoras());
		}
		return totalNoPresenciales;
	}
	
	/**
     * Se inserta el contenido de la tabla de la seccion Volumen de Trabajo
     * */
	private void contenidoTablaVolumenTrabajo(Table tabla, String contenido1, String contenido2, String contenido3, Color fondoColor, PdfFont fuente, boolean filaTotal){
		Cell cell1 = null;
		Cell cell2 = null;	
		Cell cell3 = null;
		try{
			Paragraph p1 = new Paragraph();
			p1.add(contenido1);
			p1.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p1.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			cell1 = new Cell ();
			cell1.add(p1);			
			cell1.setFontSize(11);
			cell1.setFont(fuente);			
			cell1.setBackgroundColor(fondoColor);
			cell1.setTextAlignment(TextAlignment.LEFT);
			cell1.setWidth(260);
			
			Paragraph p2 = new Paragraph();
			p2.add(contenido2);
			p2.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p2.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			cell2 = new Cell ();
			cell2.add(p2);			
			cell2.setFontSize(11);
			cell2.setFont(fuente);
			cell2.setBackgroundColor(fondoColor);
			cell2.setTextAlignment(TextAlignment.CENTER);
			
			Paragraph p3 = new Paragraph();
			p3.add(contenido3);
			p3.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p3.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			cell3 = new Cell ();
			cell3.add(p3);			
			cell3.setFontSize(11);
			cell3.setFont(fuente);
			cell3.setBackgroundColor(fondoColor);
			cell3.setTextAlignment(TextAlignment.CENTER);
			
			if(filaTotal){
				cell1.setTextAlignment(TextAlignment.RIGHT);				
				cell3.setBorderBottom(Border.NO_BORDER);
				cell3.setBorderRight(Border.NO_BORDER);
				cell3.setBackgroundColor(null);
			}
			
			tabla.addCell(cell1);
			tabla.addCell(cell2);
			tabla.addCell(cell3);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar el contenido de la tabla de Volumen de trabajo: " + e.getMessage());
		}	
	}
	
	/**
     * Se inserta la seccion Metodologia
     * */
	private void insertMetodologia(){
		try {
			
			insertLineaBlanco();
			
			//cabecera seccion RESUMEN
		    insertCabecera (traduce(GuiaConstantes.TEXT_8_METODOLOGIA), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);
		    
			String htmlText = gd.getSbMetodologia().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
			
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && "".equals(htmlText)){
			    insertCabecera (GuiaConstantes.ENGLISH_VERSION, this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false );
				insertApartadoBlob (gd.getSbMetodologia().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL));
			} else {
				insertApartadoBlob (htmlText);
			}			
				
		} catch (Exception e){
			LOGGER.info("Error insertando Metodologia: " + e.getMessage());
		}	
	}
	
	/**
     * Se inserta la seccion Evaluacion
     * */
	private void insertEvaluacion(){
		try {
			
			insertLineaBlanco();
			
			//cabecera seccion RESUMEN
		    insertCabecera (traduce(GuiaConstantes.TEXT_9_EVALUACION), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);
		    
			String htmlText = gd.getSbEvaluacion().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
			
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && "".equals(htmlText)){
			    insertCabecera (GuiaConstantes.ENGLISH_VERSION, this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false );
				insertApartadoBlob (gd.getSbEvaluacion().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL));
			} else {
				insertApartadoBlob (htmlText);
			}			
				
		} catch (Exception e){
			LOGGER.info("Error insertando Evaluacion: " + e.getMessage());
		}	
	}
	
	private void insertReferenciasSubseccion(Referencia ref, String tipoRef) {
		if (!tipoRef.equals(ref.getTipo())) {
			if (GuiaConstantes.TIPO_SUBSECCION_BASICA.equals(ref.getTipo())) {
				insertCabecera (traduce(GuiaConstantes.TEXT_10_1_BASICAS), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);
			} else {
				insertLineaColor(this.colorCabeceraSub);
				insertLineaBlanco();							
				insertCabecera (traduce(GuiaConstantes.TEXT_10_2_COMPLEMENTARIAS), this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, false);					
			}
		}
	}
	
	/**
     * Se inserta la seccion Referencias
     * */
	private void insertReferencias () {	
		
		try {
			
			ArrayList<?> alReferencias = gd.getAlReferencias();
						
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && alReferencias.isEmpty()){
				// SE OMITE EL APARTADO EN IDIOMA INGLES
			} else {
				insertLineaBlanco();
				
				//cabecera seccion REFERENCIAS
			    insertCabecera (traduce(GuiaConstantes.TEXT_10_REFERENCIAS), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);
				
				insertLineaBlanco();						
			
				String tipoRef = "";			
				for ( int i=0; i<alReferencias.size(); i++) {
					Referencia ref = (Referencia)alReferencias.get(i);
					insertReferenciasSubseccion(ref, tipoRef);
					//Creamos la tabla
					Table tabla = new Table(2);
					tabla.setWidth(UnitValue.createPercentValue(100));
					
					//Insertamos la referencia
					insertContentSubSeccionLista(tabla, ref.getDescripcion(), TextAlignment.JUSTIFIED);
					document.add(tabla);
					
					if (i == (alReferencias.size()) - 1) {
						insertLineaColor(this.colorCabeceraSub);					
					}
					
					tipoRef = ref.getTipo();
				}				
			}				
		} catch (Exception e){
			LOGGER.info("Error insertando Referencias: " + e.getMessage());
		}			
	}
	
	/**
     * Se inserta la seccion Adenda
     * */
	private void insertAdenda () {	
		
		try {
			
			insertLineaBlanco();
			
			//cabecera seccion ADENDA
			insertCabecera (traduce(GuiaConstantes.TEXT_11_ADENDA), this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false);

			insertLineaBlanco();
			
			//informacion ADENDA 
			insertCabecera(traduce(GuiaConstantes.TEXT_11_INFO),this.colorCabeceraSub, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SUBSECCION, null, true);
			
			String htmlText = gd.getSbAdenda().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);
			
			if (ConstantesGuiaDocente.IDIOMA_INGLES.equals(idioma) && "".equals(htmlText)){
			    insertCabecera (GuiaConstantes.ENGLISH_VERSION, this.colorCabecera, this.fontCabeceraSeccion, GuiaConstantes.FONT_SIZE_CABECERA_SECCION, this.colorCabeceraSeccion, false );
				insertApartadoBlob (gd.getSbAdendaMaestro().toString().replaceAll(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL));
				
			} else {
				insertApartadoBlob (htmlText);
			}		
				
		} catch (Exception e){
			LOGGER.info("Error insertando Adenda 11: " + e.getMessage());
		}	
		
	}
	
	/**
     * Se inserta un apartado Blob
     * */
	private void insertApartadoBlob(String htmlText) {
		
		insertLineaBlanco();		
		
		String str = limpiarCaracteresUTF8(htmlText);
		
		try {	
			str = tratarSignosHtmlUnicode(str);			
			str = StringEscapeUtils.unescapeHtml(str);
			
			str = str.replace("href", "style = \"word-wrap:break-word;\" href");
			
			str = str.replace("<p", "<p style = \"word-wrap:break-word;\"");
			
			List<IElement> lst = HtmlConverter.convertToElements(str);	
			Iterator<IElement> it = lst.iterator();			
			
			while(it.hasNext()){
				insertSaltoPagina(80);
				
				IElement e = it.next();
				
				Table tabla = new Table(1);
				tabla.setWidth(UnitValue.createPercentValue(100));
				
				Cell cell = null;
				cell = new Cell ();
				
				if(e instanceof Paragraph){					
					Paragraph p = (Paragraph) e;
					p.setWidth(510);				
					p.setFixedLeading(GuiaConstantes.INTERLINEADO);					
					p.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
					
					cell.add(p);			
									
				}else {	
					cell.add((IBlockElement) e);					
				}
				
				cell.setBorder(Border.NO_BORDER);
				cell.setHorizontalAlignment(HorizontalAlignment.LEFT);
				tabla.addCell(cell);
				document.add(tabla);
				
			}			
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar el apartado Blob: " + e.getMessage());
		}
		insertLineaColor(this.colorCabecera);
	}
	
	/**
     * Se inserta una cabecera
     * */
	private void insertCabecera (String contenido, Color color, String fuente, int iFontSize, Color fontColor, boolean infoAdenda ) {
		Cell cell;				
		
		try {
			
			insertSaltoPagina(100);
			
			Table tabla = new Table(1);
			tabla.setWidth(UnitValue.createPercentValue(100));
			tabla.setMarginTop(5);
						
			Paragraph p = new Paragraph();
			p.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			p.add(contenido);
			cell = new Cell ();
			cell.add(p); 
			
			cell.setHorizontalAlignment(HorizontalAlignment.LEFT);
			cell.setBorder(Border.NO_BORDER);
			cell.setBackgroundColor(color);
			cell.setFont(PdfFontFactory.createFont(fuente));			
			cell.setFontSize(iFontSize);
			cell.setFontColor(fontColor);
			
			if(infoAdenda){
				SolidBorder borde = new SolidBorder(this.colorCabecera, 1);
				cell.setBorder(borde);				
			}
			
									
			tabla.addCell(cell);
			
			document.add(tabla);
		
		} catch (Exception e) {
			LOGGER.info("Error al insertar la cabecera: " + e.getMessage());
		}	
	}
	
	/**
     * Se inserta una etiqueta de subseccion
     * */
	private void etiquetaSubSeccion(Table tabla, String etiqueta){
		Cell cell = null;
		
		try{
			Paragraph p1 = new Paragraph();
			p1.add(traduce(etiqueta));
			p1.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p1.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			cell = new Cell ();
			cell.add(p1);			
			cell.setFontSize(11);
			cell.setFont(PdfFontFactory.createFont(StandardFonts.HELVETICA_BOLD));
			cell.setHorizontalAlignment(HorizontalAlignment.LEFT);
			cell.setBorder(Border.NO_BORDER);
			
			tabla.addCell(cell);
			
		}catch (Exception e) {
			LOGGER.info("Error al insertar la etiqueta de subseccion: " + e.getMessage());
		}
	}
	
	/**
     * Se inserta contenido de subseccion
     * */
	private void insertContentSubSeccion(Table tabla, String contenido, TextAlignment ha){			
		Cell cell = null;
		
		try {
			insertSaltoPagina(80);
			contenido = tratarSignosHtmlUnicode(contenido);				
			contenido = StringEscapeUtils.unescapeHtml(contenido);
			
			contenido = contenido.replace("href", "style = \"word-wrap:break-word;\" href");
			
			Paragraph p1 = new Paragraph();
			p1.add(traduce(contenido));
			p1.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p1.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);
			p1.setWidth(510);
			cell = new Cell ();
			cell.add(p1);			
			cell.setFontSize(11);
			cell.setFont(PdfFontFactory.createFont(StandardFonts.HELVETICA));
			cell.setTextAlignment(ha);
			cell.setBorder(Border.NO_BORDER);
			
			tabla.addCell(cell);		
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar el contenido de subseccion: " + e.getMessage());
		}		
	}
	
	/**
     * Se inserta contenido de una lista de una subseccion
     * */
	private void insertContentSubSeccionLista(Table tabla, String contenido, TextAlignment ha){			
		Cell cell1 = null;
		Cell cell2 = null;	
		
		try {
			insertSaltoPagina(80);
			
			contenido = tratarSignosHtmlUnicode(contenido);				
			contenido = StringEscapeUtils.unescapeHtml(contenido);
			
			Paragraph p1 = new Paragraph(" - ");
			p1.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p1.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);			
			cell1 = new Cell ();
			cell1.add(p1);		
			cell1.setFont(PdfFontFactory.createFont(StandardFonts.HELVETICA));
			cell1.setBorder(Border.NO_BORDER);
			
			tabla.addCell(cell1);
			
			Paragraph p2 = new Paragraph();
			p2.add(traduce(contenido));
			p2.setFixedLeading(GuiaConstantes.INTERLINEADO);
			p2.setProperty(Property.OVERFLOW_WRAP, OverflowWrapPropertyValue.BREAK_WORD);			
			cell2 = new Cell ();
			cell2.add(p2);			
			cell2.setFontSize(11);
			cell2.setFont(PdfFontFactory.createFont(StandardFonts.HELVETICA));
			cell2.setTextAlignment(ha);
			cell2.setBorder(Border.NO_BORDER);
			
			tabla.addCell(cell2);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar el contenido de subseccion con lista: " + e.getMessage());
		}		
	}
	
	private String limpiarCaracteresUTF8 (String cadena) {
		String str = "";		
		str = Utils.removeNonUtf8CompliantCharacters(cadena);
		str = str.replace(GuiaConstantes.APOSTROFE_INICIAL,GuiaConstantes.APOSTROFE_FINAL);		
		
		return str;
	}
	
	private String tratarSignosHtmlUnicode(String textoAFormatear){						
					
		String vocalSinSignoDiacritico = " ";		
		String textoFormateado = " ";
		String signoHtml = " ";
		String signoUnicode = " ";
		int posicion = 0;			
		
		try{
			if (textoAFormatear != null){	
				
				for(int i = 0; i<GuiaConstantes.signosDiacriticos.length; i++){
					
					posicion = 0;
					signoHtml = GuiaConstantes.signosDiacriticos[i][0];
					signoUnicode = GuiaConstantes.signosDiacriticos[i][1];
				
					while(posicion != -1 ){
						
						Pattern patron = Pattern.compile("(.)" + signoHtml);				
						
						Matcher m = patron.matcher(textoAFormatear);
						if(m.find()){
							vocalSinSignoDiacritico = m.group(1);
						}
						
						textoAFormatear = m.replaceFirst("&" + vocalSinSignoDiacritico + signoUnicode);							
						posicion =  textoAFormatear.indexOf(signoHtml);
						
					}						
				}
				textoFormateado = textoAFormatear;
			}
			
		}catch(Exception e){
			LOGGER.info("Error al tratar signos Html Unicode: " + e.getMessage());			
		}
		
		return textoFormateado;
	}
	
	/**
	 * @param literal: literal a traducir
	 * @return el literal traducido.
	 */
	private String traduce (String literal) {
		return InternationalizedFactory.i18nLiteral( literal, this.mapa );		
	}
	
	/**
     * Se inserta una linea en blanco
     * */
	private void insertLineaBlanco () {
		 Paragraph lineaBlanco = new Paragraph("");
		 document.add(lineaBlanco);
	}
	
	/**
     * Se inserta una linea en color
     * */
	private void insertLineaColor (Color color) {
		Cell cell;	
		
		try {		
			insertLineaBlanco();
			
			SolidBorder borde = new SolidBorder(color, 0.1f);
			
			Table tabla = new Table(1);
			tabla.setWidth(UnitValue.createPercentValue(100));
			Paragraph p = new Paragraph("");
			cell = new Cell ();
			cell.add(p);
			cell.setBorder(Border.NO_BORDER);
			cell.setBorderTop(borde);
			
			tabla.addCell(cell);				
			
			document.add(tabla);
			
		} catch (Exception e) {
			LOGGER.info("Error al insertar la linea de color: " + e.getMessage());
		}
		
	}
	
	/**
     * Se inserta un salto de pagina
     * */
	private void insertSaltoPagina(int tam) {
		LayoutArea currentArea = document.getRenderer().getCurrentArea();
		Rectangle rectangle = currentArea.getBBox();
		float verticalPosition = rectangle.getHeight();
		
		if (verticalPosition < tam){
			
			document.add(new AreaBreak(AreaBreakType.NEXT_PAGE));
			insertLineaBlanco ();
		}
	}
	
}